<?php
require_once 'db.php';

function start_secure_session()
{
    $cookieParams = session_get_cookie_params();
    session_set_cookie_params([
        'lifetime' => $cookieParams["lifetime"],
        'path' => $cookieParams["path"],
        'domain' => $cookieParams["domain"],
        'secure' => isset($_SERVER['HTTPS']),
        'httponly' => true,
        'samesite' => 'Lax'
    ]);
    session_start();
}

function register_user($data)
{
    if (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
        http_response_code(400);
        return ['success' => false, 'message' => 'Invalid email format provided.'];
    }

    $conn = get_db_connection();
    $response = [];

    $sql = "INSERT INTO nsh_users (
                full_name, email, password_hash, role_id, 
                student_phone, parent_name, parent_phone, 
                temp_address, perm_address
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)";

    $stmt = $conn->prepare($sql);

    $password_hash = password_hash($data['password'], PASSWORD_DEFAULT);
    $role_id = 1;

    $stmt->bind_param(
        "sssisssss",
        $data['fullName'],
        $data['email'],
        $password_hash,
        $role_id,
        $data['studentPhone'],
        $data['parentName'],
        $data['parentPhone'],
        $data['tempAddress'],
        $data['permAddress']
    );

    // FIX: Wrap the execution in a try...catch block
    try {
        if ($stmt->execute()) {
            $response = ['success' => true, 'message' => 'User registered successfully.'];
        } else {
            // This case might not be reached if errors are exceptions, but it's good practice.
            http_response_code(500);
            $response = ['success' => false, 'message' => 'An unknown error occurred during execution.'];
        }
    } catch (mysqli_sql_exception $e) {
        // Here we CATCH the fatal error instead of crashing.
        // The error code for a duplicate entry is 1062.
        if ($e->getCode() == 1062) {
            http_response_code(409); // 409 Conflict
            $response = ['success' => false, 'message' => 'This email address is already registered.'];
        } else {
            // For any other database exception, send a generic server error.
            http_response_code(500);
            $response = ['success' => false, 'message' => 'A database error occurred.'];
        }
    }

    $stmt->close();
    $conn->close();
    return $response;
}

// login_user and logout_user functions remain unchanged...
function login_user($data)
{
    $conn = get_db_connection();
    $response = ['success' => false, 'message' => 'Invalid email or password.'];
    $stmt = $conn->prepare("SELECT user_id, password_hash, full_name, role_id FROM nsh_users WHERE email = ?");
    $stmt->bind_param("s", $data['email']);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows == 1) {
        $user = $result->fetch_assoc();
        if (password_verify($data['password'], $user['password_hash'])) {
            session_regenerate_id(true);
            $_SESSION['user_id'] = $user['user_id'];
            $_SESSION['role_id'] = $user['role_id'];
            $_SESSION['logged_in'] = true;
            $response = ['success' => true, 'message' => 'Login successful.', 'user' => ['fullName' => $user['full_name'], 'email' => $data['email']]];
        }
    }

    $stmt->close();
    $conn->close();
    return $response;
}

function logout_user()
{
    $_SESSION = array();
    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(
            session_name(),
            '',
            time() - 42000,
            $params["path"],
            $params["domain"],
            $params["secure"],
            $params["httponly"]
        );
    }
    session_destroy();
    return ['success' => true, 'message' => 'Successfully logged out.'];
}
?>